## Flex Box Basic

In this tutorial, we will learn how to use ```mui.FlexBox``` to layout components.

We recommand users to read [this tutorial](https://css-tricks.com/snippets/css/a-guide-to-flexbox/) first.

### Flex Box Container
    
```mui.FlexBox``` is a container that can layout components in a flex box. It has the following props:

* ```flex-direction```: the direction of flex box. Can be ```row | row-reverse | column | column-reverse```. Default is ```row```.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        self.flex_items = mui.FlexBox([
            mui.Typography("Item 1").prop(border="1px solid red"),
            mui.Typography("Item 2").prop(border="1px solid blue"),
            mui.Typography("Item 3").prop(border="1px solid aqua"),
        ]).prop(flexDirection="row")

        return mui.HBox([
            mui.RadioGroup(["row", "row-reverse", "column", "column-reverse"], self._on_radio_group_changed).prop(row=False),
            self.flex_items.prop(flex=1),
        ]).prop(flex=1)

    async def _on_radio_group_changed(self, value):
        await self.flex_items.send_and_wait(self.flex_items.update_event(flexDirection=value))

```

* ```flex-wrap```: the wrap mode of flex box. Can be ```nowrap | wrap | wrap-reverse```. Default is ```nowrap```.

change ```flex-wrap``` via radio group, then move the resize bar in example to see the effect.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        self.flex_items = mui.FlexBox([
            mui.Typography("Item 1").prop(border="1px solid red"),
            mui.Typography("Item 2").prop(border="1px solid blue"),
            mui.Typography("Item 3").prop(border="1px solid aqua"),
        ]).prop(flexDirection="row")

        return mui.HBox([
            mui.RadioGroup(["nowrap", "wrap", "wrap-reverse"], self._on_radio_group_changed).prop(row=False),
            mui.Allotment([
                self.flex_items,
                mui.HBox([]),
            ]).prop(defaultSizes=[1, 1])
        ]).prop(width="400px")

    async def _on_radio_group_changed(self, value):
        await self.flex_items.send_and_wait(self.flex_items.update_event(flexWrap=value))

```

* ```justify-content```: the justify content of flex box. Can be ```flex-start | flex-end | center | space-between | space-around | space-evenly```. Default is ```flex-start```.

```Python

from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        self.flex_items = mui.FlexBox([
            mui.Typography("Item 1").prop(border="1px solid red"),
            mui.Typography("Item 2").prop(border="1px solid blue"),
            mui.Typography("Item 3").prop(border="1px solid aqua"),
        ]).prop(flexDirection="row")
        options = [
            ("flex-start", "flex-start"),
            ("flex-end", "flex-end"),
            ("center", "center"),
            ("space-between", "space-between"),
            ("space-around", "space-around"),
            ("space-evenly", "space-evenly"),
        ]
        return mui.VBox([
            mui.Select("Justify Content", options, self._on_option_changed),
            self.flex_items.prop(flex=1),
        ]).prop(flex=1)

    async def _on_option_changed(self, value):
        await self.flex_items.send_and_wait(self.flex_items.update_event(justifyContent=value))

```

* ```align-items```: the align items of flex box. Can be ```flex-start | flex-end | center | baseline | stretch```. Default is ```stretch```.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        self.flex_items = mui.FlexBox([
            mui.Typography("Item 111").prop(border="1px solid red"),
            mui.Typography("Item 22222222").prop(border="1px solid blue"),
            mui.Typography("Item 333333").prop(border="1px solid aqua"),
        ]).prop(flexDirection="column")
        options = [
            ("flex-start", "flex-start"),
            ("flex-end", "flex-end"),
            ("center", "center"),
            ("baseline", "baseline"),
            ("stretch", "stretch"),
        ]
        return mui.VBox([
            mui.Select("Align Items", options, self._on_option_changed),
            self.flex_items.prop(flex=1),
        ]).prop(flex=1)

    async def _on_option_changed(self, value):
        await self.flex_items.send_and_wait(self.flex_items.update_event(alignItems=value))

```

* ```align-content```: the align content of flex box. Can be ```flex-start | flex-end | center | space-between | space-around | stretch```. Default is ```stretch```.

**Note**: This property only takes effect on multi-line flexible containers, where flex-wrap is set to either wrap or wrap-reverse. A single-line flexible container (i.e. where flex-wrap is set to its default value, no-wrap) will not reflect align-content.


```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        self.flex_items = mui.FlexBox([
            mui.Typography(f"Item {i}").prop(border="1px solid red") for i in range(20)
        ]).prop(flexDirection="row", flexWrap="wrap")
        options = [
            ("flex-start", "flex-start"),
            ("flex-end", "flex-end"),
            ("center", "center"),
            ("space-between", "space-between"),
            ("space-around", "space-around"),
            ("stretch", "stretch"),
        ]
        return mui.VBox([
            mui.Select("Align Items", options, self._on_option_changed),
            self.flex_items.prop(flex=1),
        ]).prop(flex=1)

    async def _on_option_changed(self, value):
        await self.flex_items.send_and_wait(self.flex_items.update_event(alignContent=value))

```

### Flex Item

Flex item props control the flex behavior of the component in flex parent container.

* ```flex-grow```: the flex grow factor of flex item. Default is ```0```.


```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        self.item_grow_change = mui.Typography("Item 2").prop(border="1px solid blue", flexGrow=1)
        self.flex_items = mui.FlexBox([
            mui.Typography("Item 1").prop(border="1px solid red", flexGrow=1),
            self.item_grow_change,
            mui.Typography("Item 3").prop(border="1px solid aqua", flexGrow=1),
        ]).prop(flexDirection="column")
        return mui.VBox([
            mui.Slider(0.5, 2.0, 0.05, self._on_option_changed, init_value=1.0),
            self.flex_items.prop(flex=1),
        ]).prop(flex=1)

    async def _on_option_changed(self, value):
        await self.item_grow_change.send_and_wait(self.item_grow_change.update_event(flexGrow=value))

```

* ```flex-shrink```: the flex shrink factor of flex item. Default is ```1```.


* ```flex-basis```: the flex basis of flex item. Default is ```auto```.

* ```flex```: the flex shorthand of flex item. Default is ```0 1 auto```.

```Python
from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        self.item_grow_change = mui.Typography("Item 2").prop(border="1px solid blue", flex=1)
        self.flex_items = mui.FlexBox([
            mui.Typography("Item 1").prop(border="1px solid red", flex=1),
            self.item_grow_change,
            mui.Typography("Item 3").prop(border="1px solid aqua", flex=1),
        ]).prop(flexDirection="column")
        return mui.VBox([
            mui.Slider(0.5, 2.0, 0.05, self._on_option_changed, init_value=1.0),
            self.flex_items.prop(flex=1),
        ]).prop(flex=1)

    async def _on_option_changed(self, value):
        await self.item_grow_change.send_and_wait(self.item_grow_change.update_event(flex=value))

```

* ```align-self```: the align self of flex item. Can be ```auto | flex-start | flex-end | center | baseline | stretch```. Default is ```auto```.

this property is used to override the align items of flex parent container.

this can be used to create a chat box.

```Python

from tensorpc.flow import mui, three, plus, appctx, mark_create_layout

class App:
    @mark_create_layout
    def my_layout(self):
        self.item = mui.Typography("Item 22222222").prop(border="1px solid blue", alignSelf="flex-end")
        self.flex_items = mui.FlexBox([
            mui.Typography("Item 111").prop(border="1px solid red"),
            self.item,
            mui.Typography("Item 333333").prop(border="1px solid aqua"),
        ]).prop(flexDirection="column", alignItems="flex-start")
        options = [
            ("flex-start", "flex-start"),
            ("flex-end", "flex-end"),
            ("center", "center"),
            ("baseline", "baseline"),
            ("stretch", "stretch"),
        ]
        align_self_options = [
            ("auto", "auto"),
            *options
        ]
        return mui.VBox([
            mui.Select("Align Items", options, self._on_option_changed, init_value="flex-start").prop(padding="5px 5px"),
            mui.Select("Align Item-2 Self", align_self_options, self._on_slign_self_option_changed, init_value="flex-end").prop(padding="5px 5px"),
            self.flex_items.prop(flex=1),
        ]).prop(flex=1)

    async def _on_option_changed(self, value):
        await self.flex_items.send_and_wait(self.flex_items.update_event(alignItems=value))
    
    async def _on_slign_self_option_changed(self, value):
        await self.flex_items.send_and_wait(self.item.update_event(alignSelf=value))

```
